% Copyright (c) 2016 Giampaolo D'Alessandro
% Mathematical Sciences - University of Southampton
% Email: dales@soton.ac.uk
% 
% Permission is hereby granted, free of charge, to any person obtaining a copy
% of this software and associated documentation files (the "Software"), to deal
% in the Software without restriction, including without limitation the rights
% to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
% copies of the Software, and to permit persons to whom the Software is
% furnished to do so, subject to the following conditions:
% 
% The above copyright notice and this permission notice shall be included in
% all copies or substantial portions of the Software.
% 
% THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
% IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
% FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
% AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
% LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
% OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
% SOFTWARE.

function [theta] = Theta(tInt,thetaInit,delta_1,delta_2,alpha_psi,V)
% Theta Solves the director field equation in terms of theta on a 2D slice.
%
% Input variables
% Integration time (tInt)
% Initial condition of the director field (thetaInit)
% Coefficients in the director equation (delta_1,delta_2)
% Electrostatic birefringence (alpha_psi)
% Potential difference between the two sides of the cell (V)
% Returns: the final value of the director field on a Gauss-Lobatto grid.

% Get the number of grid points from the array of initial conditions.
nz = length(thetaInit);

% Compute the matrices for the derivatives in z.
[Dz1,~] = cheb(nz-1);
Dz2 = Dz1^2;
% Compute the weights for the integral of the derivative of the potential.
[~,w]=clencurt(nz-1);
% To produce the smallest number of outputs from the Matlab ODE integration
% routine we define a minimal tspan array.
tspan = linspace(0,tInt,3);
[~,theta_tmp] = ode15s(@EqnTheta,tspan,thetaInit(2:end-1),[],delta_1,...
  delta_2,alpha_psi,Dz1,Dz2,w,thetaInit(end),thetaInit(1),V);
theta = thetaInit;
theta(2:end-1) = theta_tmp(3,:);
end